<?php

declare(strict_types=1);

namespace Erlage\Photogram;

use Exception;
use Erlage\Photogram\Constants\SystemConstants;
use Erlage\Photogram\Data\Models\AbstractModel;
use Erlage\Photogram\Data\Tables\AbstractTable;

/*
|--------------------------------------------------------------------------
| these are sys functions used in project
|--------------------------------------------------------------------------
*/

class System
{
    /**
     * Bootstrap system. start error handling, logs(if enabled), fetch config etc.
     */
    public static function init($adminMode = false)
    {
        /*
        |--------------------------------------------------------------------------
        | Global objects:
        |--------------------------------------------------------------------------
        */

        Database :: instance() -> init();
        Request  :: instance() -> init();
        Response :: instance() -> init();

        if ($adminMode)
        {
            AdminSession::instance() -> init();
        }
        else
        {
            Session  :: instance() -> init();
        }

        Settings :: init();
    }

    /**
     * @param string $tableName Name of the table as in database
     * @return AbstractTable class-string | Access to static methods/properties only.
     *  
     * Actual return type is String but we've marked it to AbstractTable. unfortunately 
     * tools for PHP aren't as good as there are in modern languages so we've done number 
     * of things to improve analysis in multiple places in entire code base. this value is
     * a class literal(AbstractModel::class), marking it as AbstractModel enables linting
     * and method reference checking.
     */
    public static function getTableClassFromTableName(string $tableName)
    {
        return Settings::schema()[$tableName];
    }

    /**
     * @param string $tableName Name of the table as in database
     * @return AbstractModel [class-string] Access to static methods/properties only
     * 
     * Actual return type is String but we've marked it to AbstractModel. 
     * @see erl_get_abstract_table_class() for more information
     */
    public static function getModelClassFromTableName(string $tableName)
    {
        return self::getTableClassFromTableName($tableName)::getModelClassName();
    }

    /**
     * Checks whether all $items are available. Availability refers whether
     * a item is assigned to value of NA system constant. Request module
     * returns NA value for keys which aren't present in client's request
     * but are requested by the system somewhere.
     */
    public static function isAvailable(string ...$items): bool
    {
        foreach ($items as $item)
        {
            if (SystemConstants::NA === $item)
            {
                return false;
            }
        }

        return true;
    }

    /**
     * Checks whther given string is a valid JSON fromat. A good method to check 
     * JSON parsability is to actually try parsing, and that's what this method do
     */
    public static function isJson(string $jsonContent): bool
    {
        if (0 === \strlen($jsonContent))
        {
            return false;
        }

        try
        {
            \json_decode($jsonContent);

            return JSON_ERROR_NONE === \json_last_error();
        }
        catch (Exception $e)
        {
            return false;
        }
    }

    public static function isMultipartRequest(): bool
    {
        return isset($_SERVER['CONTENT_TYPE'])
        && (
            0 !== \count($_FILES)
            || self::str_contains($_SERVER['CONTENT_TYPE'], 'multipart')
        );
    }

    public static function str_starts_with(string $haystack, string $needle): bool
    {
        return \strncmp($haystack, $needle, \strlen($needle)) === 0;
    }

    public static function str_ends_with(string $haystack, string $needle): bool
    {
        return $needle === '' || $needle === \substr($haystack, -\strlen($needle));
    }

    public static function str_contains(string $haystack, string $needle): bool
    {
        return '' === $needle || false !== \strpos($haystack, $needle);
    }

    public static function isoDate()
    {
        return @\date('Y-m-d', \time());
    }

    public static function isoDateTime()
    {
        return @\date('Y-m-d H:i:s', \time());
    }
}
